importScripts('study_timer.js');
importScripts('advanced_timer.js');
// Spotify auth/control handler
importScripts('spotify.js');

// 拡張機能のインストール時またはアップデート時にウェルカムページを開く
chrome.runtime.onInstalled.addListener(function(details) {
  if (details.reason === 'install') {
    chrome.tabs.create({ url: 'welcome/welcome.html' });
  }
  
  // コンテキストメニューを作成
  chrome.contextMenus.create({
    id: "askChatGPT",
    title: "ChatGPTに質問する",
    contexts: ["selection"]
  });
});

// 状態をブロードキャストするヘルパー関数
function broadcastState(kind, payload) {
  chrome.runtime.sendMessage({
    type: 'STATUS_BROADCAST',
    kind: kind,
    payload: payload
  }).catch(err => {
    // 誰もリッスンしていない場合はエラーになるが、無視してOK
    if (err.message.includes('Could not establish connection')) {
      // リスナー不在時は無視
    } else {
      console.error('Broadcast error:', err);
    }
  });
}

// ポップアップや他スクリプトからのメッセージを処理するリスナー
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // STATUS_BROADCASTは聞き流す
  if (message.type === 'STATUS_BROADCAST') {
    return;
  }

  if (message.action === 'OPEN_POPUP') {
    chrome.action.openPopup()
      .then(() => sendResponse({ success: true }))
      .catch(error => {
        console.warn('Could not open popup programmatically:', error.message);
        sendResponse({ success: false, error: error.message });
      });
    return true; // 非同期レスポンス
  }
  
  if (message.action === 'OPEN_SETTINGS_TAB') {
    // まずポップアップを開こうとする
    chrome.action.openPopup()
      .then(() => {
        // ポップアップが開けたらメッセージを送信
        setTimeout(() => {
          chrome.runtime.sendMessage({ action: 'OPEN_SETTINGS_TAB' });
        }, 100);
        sendResponse({ success: true });
      })
      .catch(error => {
        // ポップアップが開けない場合、新しいタブで開く
        console.warn('Could not open popup, opening in new tab:', error.message);
        chrome.tabs.create({
          url: chrome.runtime.getURL('popup.html#settings')
        });
        sendResponse({ success: true });
      });
    return true; // 非同期レスポンス
  }

  if (message.action === 'OPEN_ACCOUNT_TAB') {
    // まずポップアップを開く
    chrome.action.openPopup()
      .then(() => {
        // ポップアップが開けたらメッセージを送信
        setTimeout(() => {
          chrome.runtime.sendMessage({ action: 'OPEN_ACCOUNT_TAB' });
        }, 100);
        sendResponse({ success: true });
      })
      .catch(error => {
        // ポップアップが開けない場合、新しいタブで開く（アカウントタブ）
        console.warn('Could not open popup, opening in new tab:', error.message);
        chrome.tabs.create({
          url: chrome.runtime.getURL('popup.html#account')
        });
        sendResponse({ success: true });
      });
    return true; // 非同期レスポンス
  }
  
  if (message.action === 'LOAD_HTML2CANVAS') {
    if (sender.tab && sender.tab.id) {
      chrome.scripting.executeScript({
        target: { tabId: sender.tab.id },
        files: ['html2canvas.min.js']
      })
      .then(() => sendResponse({ success: true }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    }
    return true; // 非同期レスポンス
  }
  
  // 音声再生メッセージをオフスクリーンドキュメントに転送
  if (message.type && (message.type.startsWith('play') || message.type.startsWith('stop') || message.type.startsWith('set'))) {
     if (message.type.startsWith('play') || message.type.startsWith('stop') || message.type.startsWith('set')) {
      ensureOffscreenDocument().then(() => {
        chrome.runtime.sendMessage(message)
          .then(result => {
            // offscreenからの応答を待ってからブロードキャスト
            if (result && result.newState) {
              if (message.type.includes('Lofi')) {
                broadcastState('lofi', result.newState);
              } else if (message.type.includes('Ambient')) {
                broadcastState('ambient', result.newState);
              }
            }
            sendResponse(result);
          })
          .catch(error => {
            console.error('Error forwarding message to offscreen:', error);
            sendResponse({ success: false, error: error.message });
          });
      }).catch(error => {
        console.error('Error ensuring offscreen document:', error);
        sendResponse({ success: false, error: error.message });
      });
      return true; // 非同期レスポンス
    }
  }

  // データ同期リクエスト
  if (message.action === 'syncData') {
    // データ同期ロジックを実装
    chrome.storage.local.get('jwt', (result) => {
      if (result.jwt) {
        sendResponse({ success: true, message: 'Data sync initiated' });
      } else {
        sendResponse({ success: false, error: 'Not logged in' });
      }
    });
    return true; // 非同期レスポンス
  }
});

// オフスクリーンドキュメントの存在を確保する関数
async function ensureOffscreenDocument() {
  const existingContexts = await chrome.runtime.getContexts({
    contextTypes: ['OFFSCREEN_DOCUMENT'],
    documentUrls: [chrome.runtime.getURL('offscreen.html')]
  });

  if (existingContexts.length === 0) {
    await chrome.offscreen.createDocument({
      url: chrome.runtime.getURL('offscreen.html'),
      reasons: ['AUDIO_PLAYBACK'],
      justification: 'Play notification sounds for video completion'
    });
  }
}

// コンテキストメニューがクリックされた時の処理
chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === "askChatGPT" && info.selectionText) {
    // 設定を取得してからChatGPTを開く
    chrome.storage.sync.get(['chatgptTemporaryChat'], (result) => {
      // プロンプトテンプレート
      const template = `以下の内容について、答えを直接教えずにユーザーと一緒に段階的に答えに導いてください。まず最初に必ず「どこがわかっていないか」を明確にする質問をしてください。

## 質問された内容:
---
%s

---
## 応答の方針:
1. 最初の応答は「どこがわかっていないか」を尋ねる質問のみ行う（具体的に何が分からないかを掘り下げる質問を1〜2つ）。
2. ユーザーの回答を受けて、理解度に合わせて短いヒントや小さな手順を提示する。例示や考え方の提示は可だが、解答そのものは与えない。
3. ユーザーがさらに助けを求めた場合は、必要に応じて段階的に情報を追加する（徐々に具体化する）が、最終的な完全解答はユーザーが明確に要求するまで提示しない。
4. ユーザーが自分で考えることを促すフィードバックを行う。`;

      // 選択されたテキストを%sに挿入
      const promptWithSelection = template.replace('%s', info.selectionText);
      
      // URLエンコード
      const encoded = encodeURIComponent(promptWithSelection);
      
      // 一時的なチャット設定を確認
      const temporaryChat = result.chatgptTemporaryChat || false;
      
      // ChatGPTのURLを生成
      let chatGPTUrl = `https://chat.openai.com/?q=${encoded}&hints=study`;
      
      // 一時的なチャット設定に基づいてURLを調整
      if (temporaryChat) {
        chatGPTUrl += '&temporary-chat=true';
      } else {
        chatGPTUrl += '&temporary-chat=false';
      }

      // 新しいタブでChatGPTを開く
      chrome.tabs.create({ url: chatGPTUrl });
    });
  }
});
